import os
import requests
import io
from PIL import Image, ImageDraw, ImageFont, ImageOps

# --- Configuration ---
BANNER_WIDTH = 1920
BANNER_HEIGHT = 500      # Standard Itch.io header height
OUTPUT_FILENAME = "anthro_itch_banner_final.jpg"

# The Text - Combined into one line as requested
TEXT_LINE = "Anthro Entertainment LLC (Public Domain Assets)"

# Settings
FONT_URL = "https://github.com/google/fonts/raw/main/ofl/montserrat/Montserrat-Bold.ttf"
TEXT_WIDTH_PERCENT = 0.80  # Text will take up exactly 80% of the screen width
BOTTOM_MARGIN = 40         # Space from the bottom edge

def get_google_font_bytes(url):
    print(f"Downloading font from {url}...")
    try:
        response = requests.get(url)
        response.raise_for_status()
        return response.content
    except Exception as e:
        print(f"Error downloading font: {e}")
        return None

def find_images():
    files = [f for f in os.listdir('.') if f.lower().endswith('.png')]
    return files[:3]

def get_max_font_size(text, font_bytes, target_width, draw):
    """Calculates the maximum font size that fits the target width."""
    font_size = 10
    step = 5
    
    # Fast approximation loop
    while True:
        f_io = io.BytesIO(font_bytes)
        font = ImageFont.truetype(f_io, font_size)
        
        # Measure width
        bbox = draw.textbbox((0, 0), text, font=font)
        text_width = bbox[2] - bbox[0]
        
        if text_width >= target_width:
            # We went too far, step back slightly
            return font_size - step
        
        font_size += step
        
        # Safety break
        if font_size > 400: return 400

def create_banner():
    # 1. Setup Canvas
    banner = Image.new('RGB', (BANNER_WIDTH, BANNER_HEIGHT), color=(10, 10, 10))
    
    # 2. Get Images
    image_files = find_images()
    if not image_files:
        print("No .png files found!")
        return

    # 3. Stitch Images (Full Height)
    num_images = len(image_files)
    section_width = BANNER_WIDTH // num_images
    
    print(f"Stitching {num_images} images side-by-side...")
    
    for i, filename in enumerate(image_files):
        try:
            img = Image.open(filename).convert("RGB")
            # Smart crop to fill the section
            img_fitted = ImageOps.fit(img, (section_width, BANNER_HEIGHT), method=Image.Resampling.LANCZOS)
            banner.paste(img_fitted, (i * section_width, 0))
            print(f" - Added {filename}")
        except Exception as e:
            print(f" - Error processing {filename}: {e}")

    # 4. Add Gradient Overlay (for text readability)
    # Creates a fade from Transparent -> Black at the bottom
    overlay = Image.new('RGBA', (BANNER_WIDTH, BANNER_HEIGHT), (0,0,0,0))
    draw_overlay = ImageDraw.Draw(overlay)
    
    # Start fading in at 50% height
    start_y = int(BANNER_HEIGHT * 0.5)
    for y in range(start_y, BANNER_HEIGHT):
        # Calculate opacity (0 to 220)
        progress = (y - start_y) / (BANNER_HEIGHT - start_y)
        alpha = int(220 * progress) 
        draw_overlay.line([(0, y), (BANNER_WIDTH, y)], fill=(0, 0, 0, alpha))
    
    banner.paste(overlay, (0, 0), mask=overlay)

    # 5. Calculate Font Size
    font_bytes = get_google_font_bytes(FONT_URL)
    draw = ImageDraw.Draw(banner)
    
    target_text_width = BANNER_WIDTH * TEXT_WIDTH_PERCENT
    
    if font_bytes:
        final_size = get_max_font_size(TEXT_LINE, font_bytes, target_text_width, draw)
        f_io = io.BytesIO(font_bytes)
        font = ImageFont.truetype(f_io, final_size)
        print(f"Calculated Font Size: {final_size}px")
    else:
        font = ImageFont.load_default()

    # 6. Center and Draw Text
    bbox = draw.textbbox((0, 0), TEXT_LINE, font=font)
    text_width = bbox[2] - bbox[0]
    text_height = bbox[3] - bbox[1]
    
    # X: Center horizontally
    x = (BANNER_WIDTH - text_width) / 2
    
    # Y: Align to bottom margin
    y = BANNER_HEIGHT - BOTTOM_MARGIN - text_height

    # Drop Shadow (Black)
    draw.text((x + 4, y + 4), TEXT_LINE, font=font, fill=(0, 0, 0))
    
    # Main Text (White)
    draw.text((x, y), TEXT_LINE, font=font, fill=(255, 255, 255))

    # 7. Save
    banner.save(OUTPUT_FILENAME, quality=95)
    print(f"Success! Created {OUTPUT_FILENAME}")

if __name__ == "__main__":
    create_banner()